/**
 * API客户端 - 与后端翻译服务通信
 */
class TranslationAPI {
    constructor() {
        this.baseURL = 'https://translate.worldhub.world';
        this.sessionId = null;
    }

    /**
     * 初始化API客户端
     */
    async init() {
        const config = await this.getConfig();
        if (config.apiUrl) {
            this.baseURL = config.apiUrl;
        }
        this.sessionId = await this.getSessionId();
    }

    /**
     * 获取配置
     */
    async getConfig() {
        return new Promise((resolve) => {
            if (typeof chrome !== 'undefined' && chrome.storage) {
                chrome.storage.sync.get(['apiUrl', 'defaultSourceLang', 'defaultTargetLang', 'useAI', 'aiEngine'], (items) => {
                    resolve({
                        apiUrl: items.apiUrl || 'https://translate.worldhub.world',
                        defaultSourceLang: items.defaultSourceLang || 'auto',
                        defaultTargetLang: items.defaultTargetLang || 'zh-CN',
                        useAI: items.useAI || false,
                        aiEngine: items.aiEngine || 'deepseek'
                    });
                });
            } else if (typeof browser !== 'undefined' && browser.storage) {
                browser.storage.sync.get(['apiUrl', 'defaultSourceLang', 'defaultTargetLang', 'useAI', 'aiEngine']).then((items) => {
                    resolve({
                        apiUrl: items.apiUrl || 'https://translate.worldhub.world',
                        defaultSourceLang: items.defaultSourceLang || 'auto',
                        defaultTargetLang: items.defaultTargetLang || 'zh-CN',
                        useAI: items.useAI || false,
                        aiEngine: items.aiEngine || 'deepseek'
                    });
                });
            } else {
                resolve({
                    apiUrl: 'https://translate.worldhub.world',
                    defaultSourceLang: 'auto',
                    defaultTargetLang: 'zh-CN',
                    useAI: false,
                    aiEngine: 'deepseek'
                });
            }
        });
    }

    /**
     * 获取Session ID
     */
    async getSessionId() {
        return new Promise((resolve) => {
            if (typeof chrome !== 'undefined' && chrome.storage) {
                chrome.storage.local.get(['sessionId'], (items) => {
                    resolve(items.sessionId || null);
                });
            } else if (typeof browser !== 'undefined' && browser.storage) {
                browser.storage.local.get(['sessionId']).then((items) => {
                    resolve(items.sessionId || null);
                });
            } else {
                resolve(null);
            }
        });
    }

    /**
     * 设置Session ID
     */
    async setSessionId(sessionId) {
        this.sessionId = sessionId;
        return new Promise((resolve) => {
            if (typeof chrome !== 'undefined' && chrome.storage) {
                chrome.storage.local.set({ sessionId }, resolve);
            } else if (typeof browser !== 'undefined' && browser.storage) {
                browser.storage.local.set({ sessionId }).then(resolve);
            } else {
                resolve();
            }
        });
    }

    /**
     * 获取请求头
     */
    getHeaders() {
        const headers = {
            'Content-Type': 'application/json'
        };
        if (this.sessionId) {
            headers['X-Session-ID'] = this.sessionId;
        }
        return headers;
    }

    /**
     * 发送请求
     */
    async request(url, options = {}) {
        const fullUrl = url.startsWith('http') ? url : `${this.baseURL}${url}`;
        const headers = {
            ...this.getHeaders(),
            ...options.headers
        };

        try {
            const response = await fetch(fullUrl, {
                ...options,
                headers
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || `HTTP ${response.status}`);
            }

            return await response.json();
        } catch (error) {
            console.error('API请求失败:', error);
            throw error;
        }
    }

    /**
     * 翻译文本
     */
    async translateText(text, sourceLang = 'auto', targetLang = 'zh-CN', useAI = false) {
        const config = await this.getConfig();
        const finalSourceLang = sourceLang || config.defaultSourceLang || 'auto';
        const finalTargetLang = targetLang || config.defaultTargetLang || 'zh-CN';
        const finalUseAI = useAI !== undefined ? useAI : config.useAI;
        // AI翻译默认使用DeepSeek
        const aiEngine = config.aiEngine || 'deepseek';

        return this.request('/api/translate/text', {
            method: 'POST',
            body: JSON.stringify({
                text: text.trim(),
                source_lang: finalSourceLang,
                target_lang: finalTargetLang,
                use_ai: finalUseAI,
                ai_engine: finalUseAI ? aiEngine : 'chatgpt'
            })
        });
    }

    /**
     * 获取支持的语言列表（带国旗）
     */
    async getLanguages() {
        try {
            // 首先尝试从API获取
            const result = await this.request('/api/get_languages');
            if (result.success && result.languages) {
                // 为每个语言添加国旗
                const languagesWithFlags = result.languages.map(lang => {
                    const flag = this.getLanguageFlag(lang.code, lang.name || lang.native_name);
                    return {
                        ...lang,
                        flag: flag,
                        displayName: `${flag} ${lang.native_name || lang.name}`
                    };
                });
                return {
                    success: true,
                    languages: languagesWithFlags
                };
            }
        } catch (error) {
            console.error('获取语言列表失败:', error);
        }
        
        // 如果API失败，尝试从本地文件加载
        try {
            const runtime = typeof chrome !== 'undefined' && chrome.runtime 
                ? chrome.runtime 
                : browser.runtime;
            const languagesUrl = runtime.getURL('languages.json');
            const response = await fetch(languagesUrl);
            if (response.ok) {
                const languages = await response.json();
                return {
                    success: true,
                    languages: languages.map(lang => ({
                        code: lang.code,
                        name: lang.name,
                        native_name: lang.name,
                        flag: lang.flag || '🌐',
                        displayName: `${lang.flag || '🌐'} ${lang.name}`
                    }))
                };
            }
        } catch (error) {
            console.error('加载本地语言列表失败:', error);
        }
        
        // 返回默认语言列表
        return {
            success: true,
            languages: [
                { code: 'auto', name: '自动检测', native_name: '自动检测', flag: '🌐', displayName: '🌐 自动检测' },
                { code: 'zh-CN', name: '中文（简体）', native_name: '中文（简体）', flag: '🇨🇳', displayName: '🇨🇳 中文（简体）' },
                { code: 'en', name: 'English', native_name: 'English', flag: '🇬🇧', displayName: '🇬🇧 English' },
                { code: 'ja', name: 'Japanese', native_name: '日本語', flag: '🇯🇵', displayName: '🇯🇵 日本語' },
                { code: 'ko', name: 'Korean', native_name: '한국어', flag: '🇰🇷', displayName: '🇰🇷 한국어' },
                { code: 'es', name: 'Spanish', native_name: 'Español', flag: '🇪🇸', displayName: '🇪🇸 Español' },
                { code: 'fr', name: 'French', native_name: 'Français', flag: '🇫🇷', displayName: '🇫🇷 Français' },
                { code: 'de', name: 'German', native_name: 'Deutsch', flag: '🇩🇪', displayName: '🇩🇪 Deutsch' },
                { code: 'ru', name: 'Russian', native_name: 'Русский', flag: '🇷🇺', displayName: '🇷🇺 Русский' }
            ]
        };
    }
    
    /**
     * 获取语言国旗
     */
    getLanguageFlag(code, name) {
        // 语言代码到国旗的映射
        const codeToFlag = {
            'auto': '🌐',
            'zh-CN': '🇨🇳', 'zh-TW': '🇹🇼', 'zh': '🇨🇳',
            'en': '🇬🇧', 'en-US': '🇺🇸', 'en-GB': '🇬🇧',
            'ja': '🇯🇵', 'ko': '🇰🇷', 'es': '🇪🇸', 'fr': '🇫🇷',
            'de': '🇩🇪', 'ru': '🇷🇺', 'ar': '🇸🇦', 'hi': '🇮🇳',
            'pt': '🇵🇹', 'it': '🇮🇹', 'nl': '🇳🇱', 'pl': '🇵🇱',
            'tr': '🇹🇷', 'vi': '🇻🇳', 'th': '🇹🇭', 'id': '🇮🇩',
            'ms': '🇲🇾', 'he': '🇮🇱', 'cs': '🇨🇿', 'sv': '🇸🇪',
            'da': '🇩🇰', 'fi': '🇫🇮', 'no': '🇳🇴', 'hu': '🇭🇺',
            'ro': '🇷🇴', 'el': '🇬🇷', 'bg': '🇧🇬', 'hr': '🇭🇷',
            'sk': '🇸🇰', 'sl': '🇸🇮', 'et': '🇪🇪', 'lv': '🇱🇻',
            'lt': '🇱🇹', 'uk': '🇺🇦', 'be': '🇧🇾', 'ka': '🇬🇪',
            'hy': '🇦🇲', 'az': '🇦🇿', 'kk': '🇰🇿', 'uz': '🇺🇿',
            'mn': '🇲🇳', 'my': '🇲🇲', 'km': '🇰🇭', 'lo': '🇱🇦',
            'ne': '🇳🇵', 'si': '🇱🇰', 'bn': '🇧🇩', 'ur': '🇵🇰',
            'fa': '🇮🇷', 'ps': '🇦🇫', 'ku': '🇮🇶', 'am': '🇪🇹',
            'sw': '🇹🇿', 'ha': '🇳🇬', 'yo': '🇳🇬', 'ig': '🇳🇬',
            'zu': '🇿🇦', 'af': '🇿🇦', 'xh': '🇿🇦', 'st': '🇱🇸',
            'sn': '🇿🇼', 'ny': '🇲🇼', 'rw': '🇷🇼', 'so': '🇸🇴',
            'mg': '🇲🇬', 'eo': '🌍', 'la': '🏛️', 'co': '🇫🇷',
            'cy': '🏴󠁧󠁢󠁷󠁬󠁳󠁿', 'gd': '🏴󠁧󠁢󠁳󠁣󠁴󠁿', 'is': '🇮🇸',
            'ga': '🇮🇪', 'mt': '🇲🇹', 'eu': '🇪🇸', 'ca': '🇪🇸',
            'gl': '🇪🇸', 'br': '🇫🇷', 'oc': '🇫🇷', 'haw': '🇺🇸',
            'mi': '🇳🇿', 'sm': '🇼🇸', 'to': '🇹🇴', 'fj': '🇫🇯',
            'gu': '🇮🇳', 'kn': '🇮🇳', 'ml': '🇮🇳', 'ta': '🇮🇳',
            'te': '🇮🇳', 'pa': '🇮🇳', 'or': '🇮🇳', 'as': '🇮🇳',
            'mr': '🇮🇳', 'sa': '🇮🇳', 'sd': '🇵🇰', 'bal': '🇵🇰',
            'ps': '🇦🇫', 'tk': '🇹🇲', 'ky': '🇰🇬', 'tg': '🇹🇯',
            'ug': '🇨🇳', 'bo': '🇨🇳', 'hmn': '🇨🇳', 'yue': '🇭🇰',
            'ceb': '🇵🇭', 'tl': '🇵🇭', 'ilo': '🇵🇭', 'war': '🇵🇭',
            'bcl': '🇵🇭', 'pam': '🇵🇭', 'pag': '🇵🇭', 'hil': '🇵🇭',
            'jv': '🇮🇩', 'su': '🇮🇩', 'ban': '🇮🇩', 'ace': '🇮🇩',
            'min': '🇮🇩', 'bew': '🇮🇩', 'bbc': '🇮🇩', 'bts': '🇮🇩',
            'btx': '🇮🇩', 'iba': '🇲🇾', 'ms': '🇲🇾'
        };
        
        if (codeToFlag[code]) {
            return codeToFlag[code];
        }
        
        // 如果代码不在映射中，尝试从名称匹配
        if (name) {
            const nameLower = name.toLowerCase();
            if (nameLower.includes('chinese') || nameLower.includes('中文')) {
                return '🇨🇳';
            } else if (nameLower.includes('english') || nameLower.includes('英语')) {
                return '🇬🇧';
            } else if (nameLower.includes('japanese') || nameLower.includes('日语')) {
                return '🇯🇵';
            } else if (nameLower.includes('korean') || nameLower.includes('韩语')) {
                return '🇰🇷';
            } else if (nameLower.includes('spanish') || nameLower.includes('西班牙语')) {
                return '🇪🇸';
            } else if (nameLower.includes('french') || nameLower.includes('法语')) {
                return '🇫🇷';
            } else if (nameLower.includes('german') || nameLower.includes('德语')) {
                return '🇩🇪';
            } else if (nameLower.includes('russian') || nameLower.includes('俄语')) {
                return '🇷🇺';
            }
        }
        
        return '🌐';
    }

    /**
     * 检查登录状态
     */
    async checkAuth() {
        try {
            return await this.request('/api/auth/check');
        } catch (error) {
            return { success: false, user: null };
        }
    }

    /**
     * 登录
     */
    async login(email, password) {
        const result = await this.request('/api/auth/login', {
            method: 'POST',
            body: JSON.stringify({ email, password })
        });
        if (result.success && result.session_id) {
            await this.setSessionId(result.session_id);
        }
        return result;
    }

    /**
     * 登出
     */
    async logout() {
        try {
            await this.request('/api/auth/logout', { method: 'POST' });
        } catch (error) {
            console.error('登出失败:', error);
        } finally {
            await this.setSessionId(null);
        }
    }
}

// 导出API实例
if (typeof module !== 'undefined' && module.exports) {
    module.exports = TranslationAPI;
} else {
    window.TranslationAPI = TranslationAPI;
}

