/**
 * 内容脚本 - 在网页中注入翻译功能
 * 注意：此文件为共享版本，Chrome和Firefox使用各自的content.js
 */
(function () {
    'use strict';

    // 检查是否已经注入
    if (window.worldhubTranslateInjected) {
        return;
    }
    window.worldhubTranslateInjected = true;

    // API将在Chrome/Firefox特定的content.js中动态加载
    // 此文件仅作为参考，实际使用Chrome和Firefox目录中的版本
    let translationPopup = null;
    let selectedText = '';

    // 初始化API
    api.init().catch(console.error);

    /**
     * 创建翻译弹窗
     */
    function createTranslationPopup(text, translation, x, y) {
        // 移除旧弹窗
        if (translationPopup) {
            translationPopup.remove();
        }

        const popup = document.createElement('div');
        popup.id = 'worldhub-translate-popup';
        popup.innerHTML = `
            <div class="worldhub-translate-header">
                <span class="worldhub-translate-title">翻译结果</span>
                <button class="worldhub-translate-close" aria-label="关闭">×</button>
            </div>
            <div class="worldhub-translate-content">
                <div class="worldhub-translate-original">
                    <strong>原文:</strong> ${escapeHtml(text)}
                </div>
                <div class="worldhub-translate-result">
                    <strong>译文:</strong> ${escapeHtml(translation)}
                </div>
            </div>
            <div class="worldhub-translate-actions">
                <button class="worldhub-translate-copy" data-text="${escapeHtml(translation)}">复制译文</button>
                <button class="worldhub-translate-speak" data-text="${escapeHtml(translation)}" data-lang="zh-CN">朗读</button>
            </div>
        `;

        // 设置位置
        const rect = document.body.getBoundingClientRect();
        const popupX = Math.min(x, window.innerWidth - 350);
        const popupY = Math.min(y, window.innerHeight - 200);

        popup.style.left = `${popupX}px`;
        popup.style.top = `${popupY}px`;

        document.body.appendChild(popup);
        translationPopup = popup;

        // 绑定事件
        popup.querySelector('.worldhub-translate-close').addEventListener('click', () => {
            popup.remove();
            translationPopup = null;
        });

        popup.querySelector('.worldhub-translate-copy').addEventListener('click', (e) => {
            const text = e.target.getAttribute('data-text');
            copyToClipboard(text);
            showNotification('已复制到剪贴板');
        });

        popup.querySelector('.worldhub-translate-speak').addEventListener('click', (e) => {
            const text = e.target.getAttribute('data-text');
            const lang = e.target.getAttribute('data-lang') || 'zh-CN';
            speakText(text, lang);
        });

        // 点击外部关闭
        setTimeout(() => {
            document.addEventListener('click', function closePopup(e) {
                if (!popup.contains(e.target) && e.target !== popup) {
                    popup.remove();
                    translationPopup = null;
                    document.removeEventListener('click', closePopup);
                }
            }, { once: true });
        }, 100);
    }

    /**
     * 翻译选中的文本
     */
    async function translateSelectedText() {
        const text = window.getSelection().toString().trim();
        if (!text) {
            return;
        }

        selectedText = text;
        const selection = window.getSelection();
        const range = selection.getRangeAt(0);
        const rect = range.getBoundingClientRect();

        // 显示加载提示
        showLoadingIndicator(rect.left + rect.width / 2, rect.top - 10);

        try {
            const config = await api.getConfig();
            const result = await api.translateText(
                text,
                config.defaultSourceLang,
                config.defaultTargetLang,
                config.useAI
            );

            if (result.success) {
                createTranslationPopup(text, result.translated_text, rect.left, rect.top - 150);
            } else {
                showNotification('翻译失败: ' + (result.error || '未知错误'), 'error');
            }
        } catch (error) {
            console.error('翻译错误:', error);
            showNotification('翻译失败: ' + error.message, 'error');
        } finally {
            hideLoadingIndicator();
        }
    }

    /**
     * 显示加载指示器
     */
    function showLoadingIndicator(x, y) {
        const indicator = document.createElement('div');
        indicator.id = 'worldhub-translate-loading';
        indicator.innerHTML = '<div class="worldhub-translate-spinner"></div><span>翻译中...</span>';
        indicator.style.left = `${x}px`;
        indicator.style.top = `${y}px`;
        document.body.appendChild(indicator);
    }

    /**
     * 隐藏加载指示器
     */
    function hideLoadingIndicator() {
        const indicator = document.getElementById('worldhub-translate-loading');
        if (indicator) {
            indicator.remove();
        }
    }

    /**
     * 显示通知
     */
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `worldhub-translate-notification worldhub-translate-notification-${type}`;
        notification.textContent = message;
        document.body.appendChild(notification);

        setTimeout(() => {
            notification.classList.add('worldhub-translate-notification-show');
        }, 10);

        setTimeout(() => {
            notification.classList.remove('worldhub-translate-notification-show');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }

    /**
     * 复制到剪贴板
     */
    async function copyToClipboard(text) {
        try {
            if (navigator.clipboard && navigator.clipboard.writeText) {
                await navigator.clipboard.writeText(text);
            } else {
                // 降级方案
                const textarea = document.createElement('textarea');
                textarea.value = text;
                textarea.style.position = 'fixed';
                textarea.style.opacity = '0';
                document.body.appendChild(textarea);
                textarea.select();
                document.execCommand('copy');
                document.body.removeChild(textarea);
            }
        } catch (error) {
            console.error('复制失败:', error);
        }
    }

    /**
     * 朗读文本
     */
    async function speakText(text, lang) {
        try {
            const result = await api.request('/api/tts/speak', {
                method: 'POST',
                body: JSON.stringify({ text, language: lang })
            });

            if (result.success && result.audio_url) {
                const audio = new Audio(result.audio_url);
                audio.play().catch(error => {
                    console.error('播放音频失败:', error);
                    showNotification('播放失败', 'error');
                });
            } else {
                // 使用浏览器TTS作为降级方案
                if ('speechSynthesis' in window) {
                    const utterance = new SpeechSynthesisUtterance(text);
                    utterance.lang = lang;
                    speechSynthesis.speak(utterance);
                } else {
                    showNotification('不支持语音播放', 'error');
                }
            }
        } catch (error) {
            console.error('朗读失败:', error);
            // 使用浏览器TTS作为降级方案
            if ('speechSynthesis' in window) {
                const utterance = new SpeechSynthesisUtterance(text);
                utterance.lang = lang;
                speechSynthesis.speak(utterance);
            }
        }
    }

    /**
     * HTML转义
     */
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    /**
     * 翻译整个页面
     */
    async function translatePage(sourceLang, targetLang, useAI) {
        try {
            showNotification('开始翻译页面...', 'info');
            
            // 获取页面所有文本节点
            const textNodes = [];
            const walker = document.createTreeWalker(
                document.body,
                NodeFilter.SHOW_TEXT,
                null,
                false
            );
            
            let node;
            while (node = walker.nextNode()) {
                const text = node.textContent.trim();
                // 跳过空文本、脚本和样式内容
                if (text && 
                    node.parentElement && 
                    node.parentElement.tagName !== 'SCRIPT' && 
                    node.parentElement.tagName !== 'STYLE' &&
                    node.parentElement.tagName !== 'NOSCRIPT') {
                    textNodes.push({
                        node: node,
                        text: text,
                        originalText: text
                    });
                }
            }
            
            if (textNodes.length === 0) {
                showNotification('页面没有可翻译的文本', 'error');
                return { success: false, error: '页面没有可翻译的文本' };
            }
            
            showNotification(`找到 ${textNodes.length} 个文本节点，开始翻译...`, 'info');
            
            // 批量翻译（每次翻译10个节点）
            const batchSize = 10;
            let translatedCount = 0;
            
            for (let i = 0; i < textNodes.length; i += batchSize) {
                const batch = textNodes.slice(i, i + batchSize);
                const texts = batch.map(item => item.text).join('\n');
                
                try {
                    const config = await api.getConfig();
                    const result = await api.translateText(
                        texts,
                        sourceLang || config.defaultSourceLang || 'auto',
                        targetLang || config.defaultTargetLang || 'zh-CN',
                        useAI !== undefined ? useAI : config.useAI
                    );
                    
                    if (result.success) {
                        const translations = result.translated_text.split('\n');
                        batch.forEach((item, index) => {
                            if (translations[index]) {
                                item.node.textContent = translations[index];
                                translatedCount++;
                            }
                        });
                    }
                } catch (error) {
                    console.error(`翻译批次 ${i / batchSize + 1} 失败:`, error);
                }
                
                // 更新进度
                const progress = Math.round(((i + batch.length) / textNodes.length) * 100);
                showNotification(`翻译进度: ${progress}%`, 'info');
            }
            
            showNotification(`翻译完成！已翻译 ${translatedCount} 个文本节点`, 'success');
            return { success: true, translatedCount };
        } catch (error) {
            console.error('翻译页面失败:', error);
            showNotification('翻译页面失败: ' + error.message, 'error');
            return { success: false, error: error.message };
        }
    }

    /**
     * 监听消息（来自background或popup）
     */
    if (typeof chrome !== 'undefined' && chrome.runtime) {
        chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
            if (request.action === 'translate') {
                translateSelectedText().then(() => {
                    sendResponse({ success: true });
                }).catch(error => {
                    sendResponse({ success: false, error: error.message });
                });
                return true; // 异步响应
            } else if (request.action === 'translatePage') {
                translatePage(request.sourceLang, request.targetLang, request.useAI).then(result => {
                    sendResponse(result);
                }).catch(error => {
                    sendResponse({ success: false, error: error.message });
                });
                return true; // 异步响应
            }
        });
    } else if (typeof browser !== 'undefined' && browser.runtime) {
        browser.runtime.onMessage.addListener((request, sender, sendResponse) => {
            if (request.action === 'translate') {
                translateSelectedText().then(() => {
                    sendResponse({ success: true });
                }).catch(error => {
                    sendResponse({ success: false, error: error.message });
                });
                return true; // 异步响应
            } else if (request.action === 'translatePage') {
                translatePage(request.sourceLang, request.targetLang, request.useAI).then(result => {
                    sendResponse(result);
                }).catch(error => {
                    sendResponse({ success: false, error: error.message });
                });
                return true; // 异步响应
            }
        });
    }

    /**
     * 监听文本选择
     */
    let selectionTimeout = null;
    document.addEventListener('mouseup', () => {
        clearTimeout(selectionTimeout);
        selectionTimeout = setTimeout(() => {
            const text = window.getSelection().toString().trim();
            if (text && text.length > 0) {
                // 可以在这里添加自动翻译逻辑（可选）
            }
        }, 500);
    });

    /**
     * 监听双击翻译（可选功能）
     */
    document.addEventListener('dblclick', (e) => {
        const text = window.getSelection().toString().trim();
        if (text && text.length > 0) {
            translateSelectedText();
        }
    });

    /**
     * 向网页发送扩展就绪消息
     */
    function notifyPageExtensionReady() {
        try {
            // 发送消息到页面
            window.postMessage({
                type: 'worldhub-extension-ready',
                source: 'worldhub-translate-extension'
            }, '*');

            // 发送自定义事件
            const event = new CustomEvent('worldhub-extension-response', {
                detail: { type: 'pong' }
            });
            window.dispatchEvent(event);
        } catch (e) {
            console.error('发送扩展就绪消息失败:', e);
        }
    }

    // 监听来自页面的检测请求
    window.addEventListener('worldhub-extension-check', (event) => {
        if (event.detail && event.detail.type === 'ping') {
            notifyPageExtensionReady();
        }
    });

    // 页面加载完成后通知
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            setTimeout(notifyPageExtensionReady, 500);
        });
    } else {
        setTimeout(notifyPageExtensionReady, 500);
    }

    console.log('WorldHub翻译插件已加载');
})();

