/**
 * Background脚本 - 处理后台任务和消息传递
 */

// 导入API（在Chrome中使用importScripts，在Firefox中直接包含）
if (typeof importScripts !== 'undefined') {
    importScripts('api.js');
}

const api = typeof TranslationAPI !== 'undefined' ? new TranslationAPI() : null;

/**
 * 初始化扩展
 */
async function initExtension() {
    if (!api) {
        console.error('API未初始化');
        return;
    }

    await api.init();

    // 创建右键菜单
    createContextMenus();

    // 监听安装事件
    if (typeof chrome !== 'undefined' && chrome.runtime) {
        chrome.runtime.onInstalled.addListener(handleInstalled);
        chrome.runtime.onMessage.addListener(handleMessage);
    } else if (typeof browser !== 'undefined' && browser.runtime) {
        browser.runtime.onInstalled.addListener(handleInstalled);
        browser.runtime.onMessage.addListener(handleMessage);
    }
}

/**
 * 处理安装事件
 */
function handleInstalled(details) {
    if (details.reason === 'install') {
        // 首次安装，打开选项页面
        openOptionsPage();
    } else if (details.reason === 'update') {
        console.log('扩展已更新到版本', details.previousVersion);
    }
}

/**
 * 创建右键菜单
 */
function createContextMenus() {
    // 清除旧菜单
    if (typeof chrome !== 'undefined' && chrome.contextMenus) {
        chrome.contextMenus.removeAll(() => {
            chrome.contextMenus.create({
                id: 'translate-selection',
                title: '翻译选中文本',
                contexts: ['selection']
            });
        });
    } else if (typeof browser !== 'undefined' && browser.contextMenus) {
        browser.contextMenus.removeAll(() => {
            browser.contextMenus.create({
                id: 'translate-selection',
                title: '翻译选中文本',
                contexts: ['selection']
            });
        });
    }
}

/**
 * 处理消息
 */
function handleMessage(request, sender, sendResponse) {
    if (request.action === 'translate') {
        handleTranslate(request, sender, sendResponse);
        return true; // 异步响应
    } else if (request.action === 'getConfig') {
        api.getConfig().then(config => {
            sendResponse({ success: true, config });
        });
        return true;
    }
}

/**
 * 处理翻译请求
 */
async function handleTranslate(request, sender, sendResponse) {
    try {
        if (!api) {
            throw new Error('API未初始化');
        }

        const config = await api.getConfig();
        const result = await api.translateText(
            request.text,
            request.sourceLang || config.defaultSourceLang,
            request.targetLang || config.defaultTargetLang,
            request.useAI !== undefined ? request.useAI : config.useAI
        );

        sendResponse({ success: true, result });
    } catch (error) {
        console.error('翻译失败:', error);
        sendResponse({ success: false, error: error.message });
    }
}

/**
 * 处理右键菜单点击
 */
if (typeof chrome !== 'undefined' && chrome.contextMenus) {
    chrome.contextMenus.onClicked.addListener((info, tab) => {
        if (info.menuItemId === 'translate-selection' && info.selectionText) {
            // 发送消息到content script
            chrome.tabs.sendMessage(tab.id, {
                action: 'translate',
                text: info.selectionText
            });
        }
    });
} else if (typeof browser !== 'undefined' && browser.contextMenus) {
    browser.contextMenus.onClicked.addListener((info, tab) => {
        if (info.menuItemId === 'translate-selection' && info.selectionText) {
            browser.tabs.sendMessage(tab.id, {
                action: 'translate',
                text: info.selectionText
            });
        }
    });
}

/**
 * 打开选项页面
 */
function openOptionsPage() {
    const optionsUrl = typeof chrome !== 'undefined' && chrome.runtime
        ? chrome.runtime.getURL('options.html')
        : browser.runtime.getURL('options.html');

    if (typeof chrome !== 'undefined' && chrome.tabs) {
        chrome.tabs.create({ url: optionsUrl });
    } else if (typeof browser !== 'undefined' && browser.tabs) {
        browser.tabs.create({ url: optionsUrl });
    }
}

// 初始化
initExtension().catch(console.error);

