/**
 * API客户端 - 与后端翻译服务通信（WPS Office版本）
 */
class TranslationAPI {
    constructor() {
        this.baseURL = 'https://translate.worldhub.world';
        this.sessionId = null;
    }

    /**
     * 初始化API客户端
     */
    async init() {
        const config = await this.getConfig();
        if (config.apiUrl) {
            this.baseURL = config.apiUrl;
        }
        this.sessionId = await this.getSessionId();
    }

    /**
     * 获取配置（使用localStorage）
     */
    async getConfig() {
        try {
            const configStr = localStorage.getItem('worldhub_translate_config');
            if (configStr) {
                const config = JSON.parse(configStr);
                return {
                    apiUrl: config.apiUrl || 'https://translate.worldhub.world',
                    defaultSourceLang: config.defaultSourceLang || 'auto',
                    defaultTargetLang: config.defaultTargetLang || 'zh-CN',
                    useAI: config.useAI || false,
                    aiEngine: config.aiEngine || 'deepseek'
                };
            }
        } catch (error) {
            console.error('读取配置失败:', error);
        }
        return {
            apiUrl: 'https://translate.worldhub.world',
            defaultSourceLang: 'auto',
            defaultTargetLang: 'zh-CN',
            useAI: false,
            aiEngine: 'deepseek'
        };
    }

    /**
     * 保存配置
     */
    async saveConfig(config) {
        try {
            localStorage.setItem('worldhub_translate_config', JSON.stringify(config));
        } catch (error) {
            console.error('保存配置失败:', error);
        }
    }

    /**
     * 获取Session ID
     */
    async getSessionId() {
        try {
            return localStorage.getItem('worldhub_session_id') || null;
        } catch (error) {
            return null;
        }
    }

    /**
     * 设置Session ID
     */
    async setSessionId(sessionId) {
        try {
            if (sessionId) {
                localStorage.setItem('worldhub_session_id', sessionId);
            } else {
                localStorage.removeItem('worldhub_session_id');
            }
            this.sessionId = sessionId;
        } catch (error) {
            console.error('设置Session ID失败:', error);
        }
    }

    /**
     * 获取请求头
     */
    getHeaders() {
        const headers = {
            'Content-Type': 'application/json'
        };
        if (this.sessionId) {
            headers['X-Session-ID'] = this.sessionId;
        }
        return headers;
    }

    /**
     * 发送请求
     */
    async request(url, options = {}) {
        const fullUrl = url.startsWith('http') ? url : `${this.baseURL}${url}`;
        const headers = {
            ...this.getHeaders(),
            ...options.headers
        };

        try {
            const response = await fetch(fullUrl, {
                ...options,
                headers
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || `HTTP ${response.status}`);
            }

            return await response.json();
        } catch (error) {
            console.error('API请求失败:', error);
            throw error;
        }
    }

    /**
     * 翻译文本
     */
    async translateText(text, sourceLang = 'auto', targetLang = 'zh-CN', useAI = false) {
        const config = await this.getConfig();
        const finalSourceLang = sourceLang || config.defaultSourceLang || 'auto';
        const finalTargetLang = targetLang || config.defaultTargetLang || 'zh-CN';
        const finalUseAI = useAI !== undefined ? useAI : config.useAI;
        const aiEngine = config.aiEngine || 'deepseek';

        return this.request('/api/translate/text', {
            method: 'POST',
            body: JSON.stringify({
                text: text.trim(),
                source_lang: finalSourceLang,
                target_lang: finalTargetLang,
                use_ai: finalUseAI,
                ai_engine: finalUseAI ? aiEngine : 'chatgpt'
            })
        });
    }

    /**
     * 获取支持的语言列表（带国旗）
     */
    async getLanguages() {
        try {
            // 首先尝试从API获取
            const result = await this.request('/api/get_languages');
            if (result.success && result.languages) {
                const languagesWithFlags = result.languages.map(lang => {
                    const flag = this.getLanguageFlag(lang.code, lang.name || lang.native_name);
                    return {
                        ...lang,
                        flag: flag,
                        displayName: `${flag} ${lang.native_name || lang.name}`
                    };
                });
                return {
                    success: true,
                    languages: languagesWithFlags
                };
            }
        } catch (error) {
            console.error('获取语言列表失败:', error);
        }
        
        // 尝试从本地文件加载
        try {
            const response = await fetch('./languages.json');
            if (response.ok) {
                const languages = await response.json();
                return {
                    success: true,
                    languages: languages.map(lang => ({
                        code: lang.code,
                        name: lang.name,
                        native_name: lang.name,
                        flag: lang.flag || '🌐',
                        displayName: `${lang.flag || '🌐'} ${lang.name}`
                    }))
                };
            }
        } catch (error) {
            console.error('加载本地语言列表失败:', error);
        }
        
        // 返回默认语言列表
        return {
            success: true,
            languages: [
                { code: 'auto', name: '自动检测', native_name: '自动检测', flag: '🌐', displayName: '🌐 自动检测' },
                { code: 'zh-CN', name: '中文（简体）', native_name: '中文（简体）', flag: '🇨🇳', displayName: '🇨🇳 中文（简体）' },
                { code: 'en', name: 'English', native_name: 'English', flag: '🇬🇧', displayName: '🇬🇧 English' },
                { code: 'ja', name: 'Japanese', native_name: '日本語', flag: '🇯🇵', displayName: '🇯🇵 日本語' },
                { code: 'ko', name: 'Korean', native_name: '한국어', flag: '🇰🇷', displayName: '🇰🇷 한국어' }
            ]
        };
    }
    
    /**
     * 获取语言国旗
     */
    getLanguageFlag(code, name) {
        const codeToFlag = {
            'auto': '🌐', 'zh-CN': '🇨🇳', 'zh-TW': '🇹🇼', 'zh': '🇨🇳',
            'en': '🇬🇧', 'en-US': '🇺🇸', 'en-GB': '🇬🇧',
            'ja': '🇯🇵', 'ko': '🇰🇷', 'es': '🇪🇸', 'fr': '🇫🇷',
            'de': '🇩🇪', 'ru': '🇷🇺', 'ar': '🇸🇦', 'hi': '🇮🇳',
            'pt': '🇵🇹', 'it': '🇮🇹', 'nl': '🇳🇱', 'pl': '🇵🇱',
            'tr': '🇹🇷', 'vi': '🇻🇳', 'th': '🇹🇭', 'id': '🇮🇩'
        };
        
        if (codeToFlag[code]) {
            return codeToFlag[code];
        }
        
        if (name) {
            const nameLower = name.toLowerCase();
            if (nameLower.includes('chinese') || nameLower.includes('中文')) return '🇨🇳';
            if (nameLower.includes('english') || nameLower.includes('英语')) return '🇬🇧';
            if (nameLower.includes('japanese') || nameLower.includes('日语')) return '🇯🇵';
            if (nameLower.includes('korean') || nameLower.includes('韩语')) return '🇰🇷';
        }
        
        return '🌐';
    }

    /**
     * 检查登录状态
     */
    async checkAuth() {
        try {
            return await this.request('/api/auth/check');
        } catch (error) {
            return { success: false, user: null };
        }
    }

    /**
     * 登录
     */
    async login(email, password) {
        const result = await this.request('/api/auth/login', {
            method: 'POST',
            body: JSON.stringify({ email, password })
        });
        if (result.success && result.session_id) {
            await this.setSessionId(result.session_id);
        }
        return result;
    }

    /**
     * 登出
     */
    async logout() {
        try {
            await this.request('/api/auth/logout', { method: 'POST' });
        } catch (error) {
            console.error('登出失败:', error);
        } finally {
            await this.setSessionId(null);
        }
    }
}

// 导出API实例
if (typeof module !== 'undefined' && module.exports) {
    module.exports = TranslationAPI;
} else {
    window.TranslationAPI = TranslationAPI;
}

